/**
 * Login Page JavaScript
 * Universidad Nacional de San Martín - UNSM
 * Manejo de interacciones y validaciones del formulario de login
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeLoginForm();
});

/**
 * Inicializa todas las funcionalidades del formulario de login
 */
function initializeLoginForm() {
    const loginForm = document.getElementById('loginForm');
    const togglePassword = document.getElementById('togglePassword');
    const passwordInput = document.getElementById('password');
    const usernameInput = document.getElementById('username');

    // Toggle para mostrar/ocultar contraseña
    if (togglePassword && passwordInput) {
        togglePassword.addEventListener('click', function() {
            togglePasswordVisibility(passwordInput, this);
        });
    }

    // Validación en tiempo real
    if (usernameInput) {
        usernameInput.addEventListener('input', function() {
            validateUsername(this);
        });
    }

    if (passwordInput) {
        passwordInput.addEventListener('input', function() {
            validatePassword(this);
        });
    }

    // Validación al enviar el formulario
    if (loginForm) {
        loginForm.addEventListener('submit', function(e) {
            if (!validateForm()) {
                e.preventDefault();
            }
        });
    }

    // Auto-dismiss de alertas después de 5 segundos
    dismissAlerts();

    // Animación de entrada
    animateFormEntrance();
}

/**
 * Alterna la visibilidad de la contraseña
 */
function togglePasswordVisibility(passwordInput, toggleButton) {
    const type = passwordInput.type === 'password' ? 'text' : 'password';
    passwordInput.type = type;

    // Cambiar el icono del botón
    const eyeIcon = toggleButton.querySelector('.eye-icon');

    if (type === 'text') {
        eyeIcon.innerHTML = `
            <path d="M17.94 17.94A10.07 10.07 0 0 1 12 20c-7 0-11-8-11-8a18.45 18.45 0 0 1 5.06-5.94M9.9 4.24A9.12 9.12 0 0 1 12 4c7 0 11 8 11 8a18.5 18.5 0 0 1-2.16 3.19m-6.72-1.07a3 3 0 1 1-4.24-4.24"></path>
            <line x1="1" y1="1" x2="23" y2="23"></line>
        `;
        toggleButton.setAttribute('aria-label', 'Ocultar contraseña');
    } else {
        eyeIcon.innerHTML = `
            <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
            <circle cx="12" cy="12" r="3"></circle>
        `;
        toggleButton.setAttribute('aria-label', 'Mostrar contraseña');
    }
}

/**
 * Valida el campo de usuario
 */
function validateUsername(input) {
    const value = input.value.trim();
    const minLength = 3;

    removeFieldError(input);

    if (value.length > 0 && value.length < minLength) {
        showFieldError(input, `El usuario debe tener al menos ${minLength} caracteres`);
        return false;
    }

    return true;
}

/**
 * Valida el campo de contraseña
 */
function validatePassword(input) {
    const value = input.value;
    const minLength = 6;

    removeFieldError(input);

    if (value.length > 0 && value.length < minLength) {
        showFieldError(input, `La contraseña debe tener al menos ${minLength} caracteres`);
        return false;
    }

    return true;
}

/**
 * Valida todo el formulario antes de enviarlo
 */
function validateForm() {
    const usernameInput = document.getElementById('username');
    const passwordInput = document.getElementById('password');

    let isValid = true;

    // Validar usuario
    if (!usernameInput.value.trim()) {
        showFieldError(usernameInput, 'El usuario es requerido');
        isValid = false;
    } else if (!validateUsername(usernameInput)) {
        isValid = false;
    }

    // Validar contraseña
    if (!passwordInput.value) {
        showFieldError(passwordInput, 'La contraseña es requerida');
        isValid = false;
    } else if (!validatePassword(passwordInput)) {
        isValid = false;
    }

    return isValid;
}

/**
 * Muestra un error en un campo específico
 */
function showFieldError(input, message) {
    const formGroup = input.closest('.form-group');

    // Eliminar error anterior si existe
    removeFieldError(input);

    // Agregar clase de error al input
    input.classList.add('input-error');

    // Crear y agregar mensaje de error
    const errorDiv = document.createElement('div');
    errorDiv.className = 'field-error-message';
    errorDiv.textContent = message;

    formGroup.appendChild(errorDiv);

    // Agregar estilos si no existen
    if (!document.getElementById('validation-styles')) {
        const style = document.createElement('style');
        style.id = 'validation-styles';
        style.textContent = `
            .input-error {
                border-color: #dc2626 !important;
            }
            .input-error:focus {
                box-shadow: 0 0 0 3px rgba(220, 38, 38, 0.1) !important;
            }
            .field-error-message {
                color: #dc2626;
                font-size: 0.85rem;
                margin-top: 6px;
                display: flex;
                align-items: center;
                gap: 6px;
            }
            .field-error-message::before {
                content: '⚠';
            }
        `;
        document.head.appendChild(style);
    }
}

/**
 * Elimina el error de un campo
 */
function removeFieldError(input) {
    const formGroup = input.closest('.form-group');
    const errorMessage = formGroup.querySelector('.field-error-message');

    input.classList.remove('input-error');

    if (errorMessage) {
        errorMessage.remove();
    }
}

/**
 * Auto-dismiss de alertas después de un tiempo
 */
function dismissAlerts() {
    const alerts = document.querySelectorAll('.alert');

    alerts.forEach(alert => {
        setTimeout(() => {
            alert.style.transition = 'opacity 0.3s ease-out, transform 0.3s ease-out';
            alert.style.opacity = '0';
            alert.style.transform = 'translateY(-10px)';

            setTimeout(() => {
                alert.remove();
            }, 300);
        }, 5000);
    });
}

/**
 * Animación de entrada del formulario
 */
function animateFormEntrance() {
    const formContainer = document.querySelector('.login-form-container');

    if (formContainer) {
        formContainer.style.opacity = '0';
        formContainer.style.transform = 'translateY(20px)';

        setTimeout(() => {
            formContainer.style.transition = 'opacity 0.5s ease-out, transform 0.5s ease-out';
            formContainer.style.opacity = '1';
            formContainer.style.transform = 'translateY(0)';
        }, 100);
    }
}

/**
 * Muestra un indicador de carga en el botón de submit
 */
function showLoadingState(button) {
    const originalText = button.innerHTML;
    button.disabled = true;
    button.innerHTML = `
        <svg class="spinner" width="20" height="20" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
            <circle cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4" fill="none" opacity="0.25"/>
            <path d="M12 2a10 10 0 0 1 10 10" stroke="currentColor" stroke-width="4" fill="none" stroke-linecap="round"/>
        </svg>
        <span>Iniciando sesión...</span>
    `;

    // Agregar estilos del spinner si no existen
    if (!document.getElementById('spinner-styles')) {
        const style = document.createElement('style');
        style.id = 'spinner-styles';
        style.textContent = `
            @keyframes spin {
                from { transform: rotate(0deg); }
                to { transform: rotate(360deg); }
            }
            .spinner {
                animation: spin 1s linear infinite;
            }
        `;
        document.head.appendChild(style);
    }

    return originalText;
}

/**
 * Restaura el estado normal del botón
 */
function hideLoadingState(button, originalText) {
    button.disabled = false;
    button.innerHTML = originalText;
}

/**
 * Prevenir múltiples envíos del formulario
 */
(function preventMultipleSubmits() {
    const loginForm = document.getElementById('loginForm');

    if (loginForm) {
        let isSubmitting = false;

        loginForm.addEventListener('submit', function(e) {
            if (isSubmitting) {
                e.preventDefault();
                return false;
            }

            const submitButton = this.querySelector('button[type="submit"]');

            if (validateForm()) {
                isSubmitting = true;
                showLoadingState(submitButton);
            }
        });
    }
})();

/**
 * Manejo de Enter en los campos de texto
 */
document.addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        const activeElement = document.activeElement;

        if (activeElement.id === 'username') {
            e.preventDefault();
            document.getElementById('password').focus();
        }
    }
});
