<?php
/**
 * Funciones y definiciones del tema UNSM
 *
 * @package UNSM_Theme
 * @since 1.0.0
 */

// Salir si se accede directamente
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Configuración del tema
 */
function unsm_theme_setup() {
    // Soporte para título dinámico
    add_theme_support( 'title-tag' );

    // Soporte para imágenes destacadas
    add_theme_support( 'post-thumbnails' );

    // Soporte para HTML5
    add_theme_support( 'html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
        'style',
        'script',
    ) );

    // Soporte para logos personalizados
    add_theme_support( 'custom-logo', array(
        'height'      => 100,
        'width'       => 400,
        'flex-height' => true,
        'flex-width'  => true,
    ) );

    // Registrar menús de navegación
    register_nav_menus( array(
        'primary' => __( 'Menú Principal', 'unsm-theme' ),
        'footer'  => __( 'Menú Footer', 'unsm-theme' ),
    ) );

    // Soporte para actualización selectiva del Customizer
    add_theme_support( 'customize-selective-refresh-widgets' );
}
add_action( 'after_setup_theme', 'unsm_theme_setup' );

/**
 * Establecer el ancho de contenido
 */
function unsm_content_width() {
    $GLOBALS['content_width'] = apply_filters( 'unsm_content_width', 1400 );
}
add_action( 'after_setup_theme', 'unsm_content_width', 0 );

/**
 * Encolar estilos y scripts
 */
function unsm_enqueue_scripts() {
    // Estilo principal del tema
    wp_enqueue_style(
        'unsm-style',
        get_stylesheet_uri(),
        array(),
        wp_get_theme()->get( 'Version' )
    );

    // Script principal
    $script_path = get_template_directory() . '/assets/js/script.js';
    if ( file_exists( $script_path ) ) {
        wp_enqueue_script(
            'unsm-script',
            get_template_directory_uri() . '/assets/js/script.js',
            array(),
            wp_get_theme()->get( 'Version' ),
            true
        );
    }

    // Si existe un archivo js/script.js en la raíz (del sitio original)
    $legacy_script = get_template_directory() . '/js/script.js';
    if ( file_exists( $legacy_script ) ) {
        wp_enqueue_script(
            'unsm-legacy-script',
            get_template_directory_uri() . '/js/script.js',
            array(),
            wp_get_theme()->get( 'Version' ),
            true
        );
    }
}
add_action( 'wp_enqueue_scripts', 'unsm_enqueue_scripts' );

/**
 * Registrar widgets
 */
function unsm_widgets_init() {
    register_sidebar( array(
        'name'          => __( 'Sidebar Principal', 'unsm-theme' ),
        'id'            => 'sidebar-1',
        'description'   => __( 'Agrega widgets aquí para que aparezcan en tu sidebar.', 'unsm-theme' ),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h2 class="widget-title">',
        'after_title'   => '</h2>',
    ) );

    register_sidebar( array(
        'name'          => __( 'Footer 1', 'unsm-theme' ),
        'id'            => 'footer-1',
        'description'   => __( 'Área de widgets para el footer - Columna 1', 'unsm-theme' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'name'          => __( 'Footer 2', 'unsm-theme' ),
        'id'            => 'footer-2',
        'description'   => __( 'Área de widgets para el footer - Columna 2', 'unsm-theme' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );

    register_sidebar( array(
        'name'          => __( 'Footer 3', 'unsm-theme' ),
        'id'            => 'footer-3',
        'description'   => __( 'Área de widgets para el footer - Columna 3', 'unsm-theme' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ) );
}
add_action( 'widgets_init', 'unsm_widgets_init' );

/**
 * Obtener la URL base del sitio estático (compatible con instalación en subdirectorio)
 */
function unsm_get_base_url() {
    // WordPress está en /unsm/wordpress/, los archivos estáticos están en /unsm/
    // Obtener el protocolo y host
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];

    // Retornar la URL base del proyecto (un nivel arriba de WordPress)
    return $protocol . '://' . $host . '/unsm';
}

/**
 * Personalizar el excerpt
 */
function unsm_excerpt_length( $length ) {
    return 20;
}
add_filter( 'excerpt_length', 'unsm_excerpt_length', 999 );

function unsm_excerpt_more( $more ) {
    return '...';
}
add_filter( 'excerpt_more', 'unsm_excerpt_more' );

/**
 * Añadir clases personalizadas al body
 */
function unsm_body_classes( $classes ) {
    // Añadir clase si no hay sidebar
    if ( ! is_active_sidebar( 'sidebar-1' ) ) {
        $classes[] = 'no-sidebar';
    }

    return $classes;
}
add_filter( 'body_class', 'unsm_body_classes' );

/**
 * Función helper para obtener rutas de assets
 */
function unsm_asset_url( $path ) {
    // Primero buscar en el tema de WordPress
    $theme_path = get_template_directory() . '/' . ltrim( $path, '/' );
    if ( file_exists( $theme_path ) ) {
        return get_template_directory_uri() . '/' . ltrim( $path, '/' );
    }

    // Fallback a la ruta original del proyecto
    $base_url = unsm_get_base_url();
    return $base_url . '/' . ltrim( $path, '/' );
}

/**
 * Cargar archivos de imagen con fallback
 */
function unsm_get_image_url( $path ) {
    return unsm_asset_url( $path );
}

/**
 * Configurar tamaños de imagen personalizados
 */
function unsm_image_sizes() {
    add_image_size( 'unsm-carousel', 1920, 600, true );
    add_image_size( 'unsm-card', 600, 400, true );
    add_image_size( 'unsm-thumbnail', 300, 200, true );
}
add_action( 'after_setup_theme', 'unsm_image_sizes' );

/**
 * Deshabilitar la barra de administración en el frontend para usuarios no administradores
 */
function unsm_disable_admin_bar() {
    if ( ! current_user_can( 'administrator' ) && ! is_admin() ) {
        show_admin_bar( false );
    }
}
add_action( 'after_setup_theme', 'unsm_disable_admin_bar' );

/**
 * Función para mostrar el menú de navegación
 */
function unsm_navigation_menu() {
    wp_nav_menu( array(
        'theme_location' => 'primary',
        'menu_class'     => 'nav-menu',
        'container'      => false,
        'fallback_cb'    => 'unsm_default_menu',
    ) );
}

/**
 * Menú por defecto si no hay menú asignado
 */
function unsm_default_menu() {
    echo '<ul class="nav-menu">';
    wp_list_pages( array(
        'title_li' => '',
        'depth'    => 2,
    ) );
    echo '</ul>';
}

/**
 * Agregar soporte para Facebook SDK (si es necesario)
 */
function unsm_facebook_sdk() {
    ?>
    <div id="fb-root"></div>
    <script async defer crossorigin="anonymous" src="https://connect.facebook.net/es_LA/sdk.js#xfbml=1&version=v18.0"></script>
    <?php
}
add_action( 'wp_footer', 'unsm_facebook_sdk' );

/**
 * Limpieza del head
 */
remove_action( 'wp_head', 'wp_generator' );
remove_action( 'wp_head', 'wlwmanifest_link' );
remove_action( 'wp_head', 'rsd_link' );

/**
 * Permitir SVG en la biblioteca de medios
 */
function unsm_mime_types( $mimes ) {
    $mimes['svg'] = 'image/svg+xml';
    return $mimes;
}
add_filter( 'upload_mimes', 'unsm_mime_types' );

/**
 * Personalización de WordPress - Customizer API
 */
function unsm_customize_register( $wp_customize ) {
    // Sección de información de contacto
    $wp_customize->add_section( 'unsm_contact_info', array(
        'title'    => __( 'Información de Contacto', 'unsm-theme' ),
        'priority' => 30,
    ) );

    // Teléfono
    $wp_customize->add_setting( 'unsm_phone', array(
        'default'           => '(+51) (042) 48 0159',
        'sanitize_callback' => 'sanitize_text_field',
    ) );

    $wp_customize->add_control( 'unsm_phone', array(
        'label'   => __( 'Teléfono', 'unsm-theme' ),
        'section' => 'unsm_contact_info',
        'type'    => 'text',
    ) );

    // Email
    $wp_customize->add_setting( 'unsm_email', array(
        'default'           => 'informes@unsm.edu.pe',
        'sanitize_callback' => 'sanitize_email',
    ) );

    $wp_customize->add_control( 'unsm_email', array(
        'label'   => __( 'Email', 'unsm-theme' ),
        'section' => 'unsm_contact_info',
        'type'    => 'email',
    ) );

    // Dirección
    $wp_customize->add_setting( 'unsm_address', array(
        'default'           => 'Jr. Maynas N° 177 - Morales, Tarapoto - San Martín - Perú',
        'sanitize_callback' => 'sanitize_textarea_field',
    ) );

    $wp_customize->add_control( 'unsm_address', array(
        'label'   => __( 'Dirección', 'unsm-theme' ),
        'section' => 'unsm_contact_info',
        'type'    => 'textarea',
    ) );
}
add_action( 'customize_register', 'unsm_customize_register' );