'use strict';

/**
 * Objeto Rectorado UNSM
 * Maneja la funcionalidad específica de la página de Rectorado
 */
const Rectorado = {
    // ============================================
    // ESTADO DE LA APLICACIÓN
    // ============================================
    state: {
        currentTab: 'funciones',
        currentPage: 1,
        expandedItems: new Set()
    },

    // ============================================
    // INICIALIZACIÓN
    // ============================================
    init() {
        try {
            console.log('========================================');
            console.log('🏛️ INICIANDO MÓDULO RECTORADO');
            console.log('========================================');

            this.initTabs();
            console.log('✅ Sistema de tabs inicializado');

            // Las funciones de agenda se manejan desde WordPress
            // this.initReadMore();
            // this.initPagination();

            console.log('========================================');
            console.log('✅ MÓDULO RECTORADO CARGADO');
            console.log('========================================');
        } catch (error) {
            console.error('❌ Error durante la inicialización de Rectorado:', error);
        }
    },

    // ============================================
    // SISTEMA DE TABS
    // ============================================
    initTabs() {
        const tabButtons = document.querySelectorAll('.tab-button');
        const tabContents = document.querySelectorAll('.tab-content');

        if (tabButtons.length === 0) {
            console.warn('⚠️ No se encontraron botones de tabs');
            return;
        }

        tabButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                const targetTab = button.getAttribute('data-tab');
                this.switchTab(targetTab, tabButtons, tabContents);
            });
        });
    },

    switchTab(targetTab, tabButtons, tabContents) {
        // Actualizar estado
        this.state.currentTab = targetTab;

        // Remover clase active de todos los botones
        tabButtons.forEach(btn => {
            btn.classList.remove('active');
            btn.setAttribute('aria-selected', 'false');
        });

        // Remover clase active de todos los contenidos
        tabContents.forEach(content => {
            content.classList.remove('active');
        });

        // Agregar clase active al botón seleccionado
        const activeButton = document.querySelector(`[data-tab="${targetTab}"]`);
        if (activeButton) {
            activeButton.classList.add('active');
            activeButton.setAttribute('aria-selected', 'true');
        }

        // Agregar clase active al contenido correspondiente
        const activeContent = document.getElementById(targetTab);
        if (activeContent) {
            activeContent.classList.add('active');
        }

        console.log(`📑 Tab cambiado a: ${targetTab}`);
    },

    // ============================================
    // BOTONES "LEER MÁS" - WordPress Integration
    // ============================================
    // Esta función estará disponible para cuando WordPress agregue contenido dinámicamente
    initReadMore() {
        const readMoreButtons = document.querySelectorAll('.btn-leer-mas');

        if (readMoreButtons.length === 0) {
            console.warn('⚠️ No se encontraron botones "Leer más"');
            return;
        }

        readMoreButtons.forEach((button, index) => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                this.toggleReadMore(button, index);
            });
        });
    },

    toggleReadMore(button, index) {
        const agendaItem = button.closest('.agenda-item');
        if (!agendaItem) return;

        const desc = agendaItem.querySelector('.agenda-desc');

        // Si el item está expandido, contraerlo
        if (this.state.expandedItems.has(index)) {
            agendaItem.classList.remove('expanded');
            button.textContent = 'Leer más';
            this.state.expandedItems.delete(index);

            if (desc && !desc.style.display) {
                desc.style.display = 'none';
            }
        } else {
            // Si el item está contraído, expandirlo
            agendaItem.classList.add('expanded');
            button.textContent = 'Leer menos';
            this.state.expandedItems.add(index);

            if (desc) {
                desc.style.display = 'block';
            }
        }

        console.log(`📖 Item ${index} ${this.state.expandedItems.has(index) ? 'expandido' : 'contraído'}`);
    },

    // ============================================
    // PAGINACIÓN - WordPress Integration
    // ============================================
    // Esta función estará disponible para cuando WordPress agregue paginación
    initPagination() {
        const pagination = document.querySelector('.pagination');
        if (!pagination) {
            console.warn('⚠️ No se encontró sistema de paginación');
            return;
        }

        const pageNumbers = pagination.querySelectorAll('.page-number');

        pageNumbers.forEach((page, index) => {
            page.addEventListener('click', (e) => {
                e.preventDefault();

                // Extraer número de página del texto
                const pageText = page.textContent.trim();

                if (pageText.includes('Siguiente')) {
                    this.goToNextPage();
                } else if (pageText.includes('Anterior')) {
                    this.goToPreviousPage();
                } else {
                    const pageNum = parseInt(pageText);
                    if (!isNaN(pageNum)) {
                        this.goToPage(pageNum);
                    }
                }
            });

            // Agregar estilos de cursor para mejor UX
            page.style.cursor = 'pointer';
        });
    },

    goToPage(pageNum) {
        this.state.currentPage = pageNum;
        console.log(`📄 Navegando a página ${pageNum}`);

        // Actualizar estado visual de la paginación
        this.updatePaginationUI();

        // Scroll suave hacia arriba de la agenda
        const agendaSection = document.querySelector('.agenda-section');
        if (agendaSection) {
            agendaSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
        }

        // Aquí se podría implementar la carga AJAX de nuevos items
        // this.loadAgendaItems(pageNum);
    },

    goToNextPage() {
        this.goToPage(this.state.currentPage + 1);
    },

    goToPreviousPage() {
        if (this.state.currentPage > 1) {
            this.goToPage(this.state.currentPage - 1);
        }
    },

    updatePaginationUI() {
        const pageNumbers = document.querySelectorAll('.page-number');

        pageNumbers.forEach(page => {
            const pageText = page.textContent.trim();
            const pageNum = parseInt(pageText);

            if (!isNaN(pageNum)) {
                if (pageNum === this.state.currentPage) {
                    page.style.fontWeight = 'bold';
                    page.style.color = '#359444';
                    page.setAttribute('aria-current', 'page');
                } else {
                    page.style.fontWeight = 'normal';
                    page.style.color = '#333';
                    page.removeAttribute('aria-current');
                }
            }
        });
    },

    // ============================================
    // UTILIDADES PARA WORDPRESS
    // ============================================

    /**
     * Método para cargar items de agenda dinámicamente desde WordPress
     * @param {number} page - Número de página a cargar
     */
    loadAgendaItems(page) {
        console.log(`🔄 Cargando items de la página ${page} desde WordPress...`);

        // Este método se puede usar para cargar contenido desde WordPress vía AJAX
        // Ejemplo de implementación:
        /*
        fetch(`/wp-json/unsm/v1/rectorado/agenda?page=${page}`)
            .then(response => response.json())
            .then(data => {
                this.renderAgendaItems(data.items);
                this.initReadMore();
            })
            .catch(error => {
                console.error('Error al cargar items:', error);
            });
        */
    },

    /**
     * Método para renderizar items de agenda
     * @param {Array} items - Array de items de agenda
     */
    renderAgendaItems(items) {
        const agendaItemsContainer = document.querySelector('.agenda-items');
        if (!agendaItemsContainer) return;

        // Limpiar items actuales
        agendaItemsContainer.innerHTML = '';

        // Renderizar nuevos items
        items.forEach(item => {
            const article = this.createAgendaItem(item);
            agendaItemsContainer.appendChild(article);
        });
    },

    /**
     * Método para crear un elemento de agenda
     * @param {Object} item - Objeto con datos del item de agenda
     */
    createAgendaItem(item) {
        const article = document.createElement('article');
        article.className = 'agenda-item';

        article.innerHTML = `
            <h4>${item.title}</h4>
            <p class="agenda-date">${item.date}</p>
            <p class="agenda-time"><strong>Cuando:</strong> ${item.time}</p>
            ${item.description ? `<p class="agenda-desc">${item.description}</p>` : ''}
            <button class="btn-leer-mas">Leer más</button>
        `;

        return article;
    }
};

// ============================================
// INICIALIZACIÓN AL CARGAR EL DOM
// ============================================
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => Rectorado.init());
} else {
    Rectorado.init();
}

// ============================================
// MANEJO DE ERRORES ESPECÍFICO DEL MÓDULO
// ============================================
window.addEventListener('error', (event) => {
    if (event.filename && event.filename.includes('srciptRecto.js')) {
        console.error('❌ Error en módulo Rectorado:', event.error);
    }
});

// ============================================
// EXPORTAR PARA USO EN MÓDULOS (OPCIONAL)
// ============================================
if (typeof module !== 'undefined' && module.exports) {
    module.exports = Rectorado;
}
