// Script para la página del Himno Institucional

document.addEventListener('DOMContentLoaded', function() {
    // Smooth scroll para enlaces internos
    const links = document.querySelectorAll('a[href^="#"]');
    
    links.forEach(link => {
        link.addEventListener('click', function(e) {
            const href = this.getAttribute('href');
            if (href !== '#' && href.length > 1) {
                const target = document.querySelector(href);
                if (target) {
                    e.preventDefault();
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            }
        });
    });
    
    // Animación de entrada para el contenido
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver(function(entries) {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.style.opacity = '1';
                entry.target.style.transform = 'translateY(0)';
            }
        });
    }, observerOptions);
    
    const animatedElements = document.querySelectorAll('.himno-stanza, .himno-coro, .himno-credits');
    animatedElements.forEach(el => {
        el.style.opacity = '0';
        el.style.transform = 'translateY(20px)';
        el.style.transition = 'opacity 0.6s ease, transform 0.6s ease';
        observer.observe(el);
    });

    // Inicializar reproductor de audio
    initAudioPlayer();
});

/**
 * Inicializa el reproductor de audio del himno
 */
function initAudioPlayer() {
    const audio = document.getElementById('himnoAudio');
    const playBtn = document.getElementById('playBtn');
    const playIcon = playBtn.querySelector('.play-icon');
    const progressBar = document.getElementById('progressBar');
    const progressFill = document.getElementById('progressFill');
    const currentTimeEl = document.getElementById('currentTime');
    const durationEl = document.getElementById('duration');
    const volumeBtn = document.getElementById('volumeBtn');
    const volumeSlider = document.getElementById('volumeSlider');
    const volumeIcon = volumeBtn.querySelector('.volume-icon');

    // Verificar que el audio existe
    if (!audio) return;

    // Configurar volumen inicial
    audio.volume = volumeSlider.value / 100;

    // Cargar metadata del audio
    audio.addEventListener('loadedmetadata', function() {
        durationEl.textContent = formatTime(audio.duration);
    });

    // Boton de play/pause
    playBtn.addEventListener('click', function() {
        if (audio.paused) {
            audio.play();
            playIcon.innerHTML = '&#10074;&#10074;'; // Icono de pausa
            playBtn.classList.add('playing');
        } else {
            audio.pause();
            playIcon.innerHTML = '&#9654;'; // Icono de play
            playBtn.classList.remove('playing');
        }
    });

    // Actualizar barra de progreso
    audio.addEventListener('timeupdate', function() {
        const percent = (audio.currentTime / audio.duration) * 100;
        progressFill.style.width = percent + '%';
        currentTimeEl.textContent = formatTime(audio.currentTime);
    });

    // Click en la barra de progreso
    progressBar.addEventListener('click', function(e) {
        const rect = progressBar.getBoundingClientRect();
        const percent = (e.clientX - rect.left) / rect.width;
        audio.currentTime = percent * audio.duration;
    });

    // Control de volumen con slider
    volumeSlider.addEventListener('input', function() {
        audio.volume = this.value / 100;
        updateVolumeIcon(this.value);
    });

    // Boton de silencio
    volumeBtn.addEventListener('click', function() {
        if (audio.volume > 0) {
            audio.dataset.lastVolume = audio.volume;
            audio.volume = 0;
            volumeSlider.value = 0;
            updateVolumeIcon(0);
        } else {
            const lastVolume = audio.dataset.lastVolume || 0.7;
            audio.volume = lastVolume;
            volumeSlider.value = lastVolume * 100;
            updateVolumeIcon(lastVolume * 100);
        }
    });

    // Cuando termina la cancion
    audio.addEventListener('ended', function() {
        playIcon.innerHTML = '&#9654;';
        playBtn.classList.remove('playing');
        progressFill.style.width = '0%';
        audio.currentTime = 0;
    });

    // Actualizar icono de volumen inicial
    updateVolumeIcon(volumeSlider.value);
}

/**
 * Formatea el tiempo en minutos:segundos
 * @param {number} seconds - Tiempo en segundos
 * @returns {string} Tiempo formateado
 */
function formatTime(seconds) {
    if (isNaN(seconds)) return '0:00';

    const mins = Math.floor(seconds / 60);
    const secs = Math.floor(seconds % 60);
    return mins + ':' + (secs < 10 ? '0' : '') + secs;
}

/**
 * Actualiza el icono del volumen segun el nivel
 * @param {number} volume - Nivel de volumen (0-100)
 */
function updateVolumeIcon(volume) {
    const volumeIcon = document.querySelector('.volume-icon');
    if (!volumeIcon) return;

    if (volume == 0) {
        volumeIcon.innerHTML = '&#128263;'; // Mute
    } else if (volume < 50) {
        volumeIcon.innerHTML = '&#128264;'; // Low volume
    } else {
        volumeIcon.innerHTML = '&#128266;'; // High volume
    }
}
