/**
 * CHATBOT UNSM - Asistente Virtual
 * Sistema de ayuda basado en reglas y palabras clave
 * Versión: 1.0.0
 */

'use strict';

// Extender el namespace UNSM existente
UNSM.Chatbot = {
    // ============================================
    // ESTADO DEL CHATBOT
    // ============================================
    state: {
        isOpen: false,
        currentPage: window.location.pathname,
        messages: [],
        lastInteraction: null,
        isTyping: false,
        userRole: null // Rol del usuario (estudiante, docente, administrativo, visitante)
    },

    // ============================================
    // INICIALIZACIÓN
    // ============================================
    init: function() {
        // Esperar a que CHATBOT_DATA esté disponible
        if (typeof CHATBOT_DATA === 'undefined') {
            console.error('CHATBOT_DATA no está definido. Asegúrate de cargar chatbot-data.js primero.');
            return;
        }

        // Cachear elementos del DOM
        this.cacheElements();

        // Vincular eventos
        this.bindEvents();

        // Cargar rol guardado del usuario (si existe)
        this.loadUserRole();

        // Cargar mensaje de bienvenida
        this.loadWelcomeMessage();

        // Configurar acciones rápidas (solo si hay rol guardado)
        if (this.state.userRole) {
            this.setQuickActions();
        } else {
            // Ocultar acciones rápidas hasta que se seleccione un rol
            if (this.elements.quickActions) {
                this.elements.quickActions.style.display = 'none';
            }
        }

        // Añadir animación de pulso al botón (solo primeros segundos)
        setTimeout(() => {
            this.elements.button.classList.add('chatbot__button--pulse');
            setTimeout(() => {
                this.elements.button.classList.remove('chatbot__button--pulse');
            }, 6000);
        }, 1000);

        console.log('✅ Chatbot UNSM inicializado correctamente');
    },

    // ============================================
    // CARGAR ROL DEL USUARIO
    // ============================================
    loadUserRole: function() {
        // NO cargar el rol automáticamente - siempre preguntar
        // El usuario debe seleccionar su rol cada vez que abre el chatbot
        this.state.userRole = null;

        // Limpiar cualquier rol guardado anteriormente
        try {
            localStorage.removeItem('unsm_chatbot_role');
        } catch (e) {
            console.warn('No se pudo limpiar el rol del localStorage');
        }

        console.log('ℹ️ Esperando selección de rol del usuario');
    },

    // ============================================
    // CACHEAR ELEMENTOS DEL DOM
    // ============================================
    cacheElements: function() {
        this.elements = {
            button: document.getElementById('chatbotButton'),
            window: document.getElementById('chatbotWindow'),
            closeBtn: document.getElementById('chatbotCloseBtn'),
            messages: document.getElementById('chatbotMessages'),
            input: document.getElementById('chatbotInput'),
            sendBtn: document.getElementById('chatbotSendBtn'),
            quickActions: document.getElementById('chatbotQuickActions'),
            typing: document.getElementById('chatbotTyping')
        };

        // Verificar que todos los elementos existen
        for (const [key, element] of Object.entries(this.elements)) {
            if (!element) {
                console.error(`Elemento no encontrado: ${key}`);
            }
        }
    },

    // ============================================
    // VINCULAR EVENTOS
    // ============================================
    bindEvents: function() {
        // Botón flotante - toggle
        this.elements.button.addEventListener('click', () => this.toggle());

        // Botón cerrar
        this.elements.closeBtn.addEventListener('click', () => this.close());

        // Botón enviar
        this.elements.sendBtn.addEventListener('click', () => this.handleSend());

        // Input - Enter para enviar
        this.elements.input.addEventListener('keypress', (e) => {
            if (e.key === 'Enter' && !e.shiftKey) {
                e.preventDefault();
                this.handleSend();
            }
        });

        // Detectar clicks en preguntas relacionadas
        this.elements.messages.addEventListener('click', (e) => {
            if (e.target.classList.contains('chatbot__related-item')) {
                const faqId = e.target.dataset.faqId;
                this.handleQuickAction(faqId);
            }
        });

        // Escape para cerrar
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && this.state.isOpen) {
                this.close();
            }
        });
    },

    // ============================================
    // CONTROLES DE UI
    // ============================================
    toggle: function() {
        this.state.isOpen ? this.close() : this.open();
    },

    open: function() {
        this.elements.window.classList.add('chatbot__window--open');
        this.elements.button.classList.add('chatbot__button--active');
        this.elements.window.setAttribute('aria-hidden', 'false');
        this.elements.button.setAttribute('aria-expanded', 'true');
        this.state.isOpen = true;

        // Enfocar el input después de la animación
        setTimeout(() => {
            this.elements.input.focus();
        }, 300);

        // Scroll al final de los mensajes
        this.scrollToBottom();
    },

    close: function() {
        this.elements.window.classList.remove('chatbot__window--open');
        this.elements.button.classList.remove('chatbot__button--active');
        this.elements.window.setAttribute('aria-hidden', 'true');
        this.elements.button.setAttribute('aria-expanded', 'false');
        this.state.isOpen = false;
    },

    // ============================================
    // MANEJO DE MENSAJES
    // ============================================
    handleSend: function() {
        const message = this.elements.input.value.trim();

        // Validar que hay mensaje
        if (!message) return;

        // Comando especial para cambiar de rol
        if (message.toLowerCase() === 'cambiar rol' || message.toLowerCase() === 'cambiar mi rol') {
            this.elements.input.value = '';
            this.addMessage(message, 'user');
            this.changeRole();
            return;
        }

        // Validar longitud máxima
        if (message.length > CHATBOT_DATA.config.messageMaxLength) {
            this.addMessage(
                `El mensaje es demasiado largo. Por favor, limítalo a ${CHATBOT_DATA.config.messageMaxLength} caracteres.`,
                'bot'
            );
            return;
        }

        // Añadir mensaje del usuario
        this.addMessage(message, 'user');

        // Limpiar input
        this.elements.input.value = '';

        // Mostrar indicador de escritura
        this.showTyping();

        // Procesar y responder
        setTimeout(() => {
            this.hideTyping();
            this.respondToMessage(message);
        }, CHATBOT_DATA.config.typingDelay);

        // Actualizar última interacción
        this.state.lastInteraction = new Date();
    },

    addMessage: function(text, sender, options = {}) {
        // Crear contenedor del mensaje
        const messageDiv = document.createElement('div');
        messageDiv.className = `chatbot__message chatbot__message--${sender}`;

        // Crear burbuja
        const bubbleDiv = document.createElement('div');
        bubbleDiv.className = 'chatbot__bubble';

        // Convertir saltos de línea a <br> de manera segura
        const formattedText = this.formatMessageText(text);
        bubbleDiv.innerHTML = formattedText;

        // Crear timestamp
        const timestamp = this.formatTimestamp(new Date());
        const timeDiv = document.createElement('div');
        timeDiv.className = 'chatbot__timestamp';
        timeDiv.textContent = timestamp;

        // Ensamblar
        messageDiv.appendChild(bubbleDiv);
        messageDiv.appendChild(timeDiv);

        // Si hay preguntas relacionadas, agregarlas
        if (options.relatedFaqs && options.relatedFaqs.length > 0) {
            const relatedDiv = this.createRelatedQuestions(options.relatedFaqs);
            bubbleDiv.appendChild(relatedDiv);
        }

        // Añadir al DOM
        this.elements.messages.appendChild(messageDiv);

        // Scroll al final
        this.scrollToBottom();

        // Guardar en estado
        this.state.messages.push({ text, sender, timestamp: new Date() });
    },

    createRelatedQuestions: function(faqIds) {
        const container = document.createElement('div');
        container.className = 'chatbot__related';

        const title = document.createElement('p');
        title.className = 'chatbot__related-title';
        title.textContent = 'Preguntas relacionadas:';
        container.appendChild(title);

        const list = document.createElement('ul');
        list.className = 'chatbot__related-list';

        // Limitar a max related questions
        const maxRelated = CHATBOT_DATA.config.maxRelatedQuestions || 3;
        const limitedIds = faqIds.slice(0, maxRelated);

        limitedIds.forEach(faqId => {
            const faq = this.findFaqById(faqId);
            if (faq) {
                const item = document.createElement('li');
                item.className = 'chatbot__related-item';
                item.textContent = faq.question;
                item.dataset.faqId = faqId;
                item.setAttribute('role', 'button');
                item.setAttribute('tabindex', '0');
                list.appendChild(item);
            }
        });

        container.appendChild(list);
        return container;
    },

    // ============================================
    // MOTOR DE RESPUESTAS
    // ============================================
    respondToMessage: function(message) {
        // Intentar encontrar coincidencia
        const result = this.matchKeywords(message);

        if (result && result.confidence >= CHATBOT_DATA.config.confidenceThreshold) {
            // Encontró coincidencia
            const faq = result.faq;
            this.addMessage(faq.answer, 'bot', {
                relatedFaqs: faq.relatedFaqs
            });
        } else {
            // No encontró coincidencia - mostrar fallback
            this.showFallbackMessage();
        }
    },

    showFallbackMessage: function() {
        const fallback = CHATBOT_DATA.fallback;

        // Mensaje principal
        this.addMessage(fallback.noMatch, 'bot');

        // Sugerencias de preguntas comunes
        if (fallback.suggestions && fallback.suggestions.length > 0) {
            const suggestionsText = 'Prueba preguntando sobre:\n\n' +
                fallback.suggestions.map((q, i) => `${i + 1}. ${q}`).join('\n');
            this.addMessage(suggestionsText, 'bot');
        }

        // Información de contacto
        if (fallback.contactInfo) {
            const contactText = `O contáctanos directamente:\n\n📧 ${fallback.contactInfo.email}\n📞 ${fallback.contactInfo.phone}`;
            this.addMessage(contactText, 'bot');
        }
    },

    // ============================================
    // MOTOR DE COINCIDENCIA DE PALABRAS CLAVE
    // ============================================
    matchKeywords: function(userInput) {
        // Normalizar input
        const normalized = this.normalizeText(userInput);

        // Tokenizar
        const tokens = this.tokenize(normalized);

        if (tokens.length === 0) return null;

        // Buscar en todos los FAQs
        const results = [];

        for (const category of Object.values(CHATBOT_DATA.categories)) {
            for (const faq of category.faqs) {
                // Contar coincidencias de keywords
                let keywordMatches = 0;
                let tokenMatches = 0;
                const matchedTokens = new Set();

                for (const keyword of faq.keywords) {
                    const normalizedKeyword = this.normalizeText(keyword);

                    // Verificar si algún token coincide con el keyword
                    for (const token of tokens) {
                        if (token.includes(normalizedKeyword) || normalizedKeyword.includes(token)) {
                            keywordMatches++;
                            matchedTokens.add(token);
                            break;
                        }
                    }
                }

                tokenMatches = matchedTokens.size;

                // Calcular confianza combinada:
                // - Porcentaje de keywords que coincidieron
                // - Porcentaje de tokens del usuario que coincidieron
                // Esto permite que frases cortas como "me gusta la medicina" funcionen
                if (keywordMatches > 0) {
                    const keywordRatio = keywordMatches / faq.keywords.length;
                    const tokenRatio = tokenMatches / tokens.length;
                    // Dar más peso al porcentaje de tokens del usuario que coincidieron
                    const confidence = (keywordRatio * 0.4) + (tokenRatio * 0.6);
                    results.push({ faq, confidence, keywordMatches });
                }
            }
        }

        // Ordenar por confianza (mayor primero), y por keywordMatches si hay empate
        results.sort((a, b) => {
            if (b.confidence !== a.confidence) return b.confidence - a.confidence;
            return b.keywordMatches - a.keywordMatches;
        });

        // Retornar mejor resultado o null
        return results.length > 0 ? results[0] : null;
    },

    normalizeText: function(text) {
        return text
            .toLowerCase()
            .normalize('NFD')
            .replace(/[\u0300-\u036f]/g, '') // Quitar acentos
            .replace(/[¿?¡!.,;:]/g, '') // Quitar puntuación
            .trim();
    },

    tokenize: function(text) {
        // Dividir en palabras y filtrar palabras muy cortas
        return text.split(/\s+/).filter(word => word.length > 2);
    },

    // ============================================
    // ACCIONES RÁPIDAS
    // ============================================
    handleQuickAction: function(faqId) {
        const faq = this.findFaqById(faqId);

        if (faq) {
            // Añadir pregunta como mensaje del usuario
            this.addMessage(faq.question, 'user');

            // Mostrar typing
            this.showTyping();

            // Responder después de delay
            setTimeout(() => {
                this.hideTyping();
                this.addMessage(faq.answer, 'bot', {
                    relatedFaqs: faq.relatedFaqs
                });
            }, 500);
        }
    },

    setQuickActions: function() {
        let actions;
        let source = 'default';

        // Si hay un rol seleccionado, usar acciones del rol
        if (this.state.userRole && CHATBOT_DATA.quickActions[this.state.userRole]) {
            actions = CHATBOT_DATA.quickActions[this.state.userRole];
            source = `rol: ${this.state.userRole}`;
        } else {
            // Si no hay rol, usar acciones de la página actual
            const page = this.detectPage();
            actions = CHATBOT_DATA.quickActions[page] || CHATBOT_DATA.quickActions.default;
            source = `página: ${page}`;
        }

        console.log(`🔧 Mostrando acciones rápidas de ${source}`, actions);
        this.renderQuickActions(actions);
    },

    renderQuickActions: function(actions) {
        this.elements.quickActions.innerHTML = '';

        if (!actions || actions.length === 0) {
            console.warn('⚠️ No hay acciones rápidas para mostrar');
            return;
        }

        console.log('🎯 Renderizando acciones rápidas:', actions);

        actions.forEach(action => {
            const button = document.createElement('button');
            button.className = 'chatbot__quick-action';
            button.textContent = action.text;
            button.dataset.faqId = action.faqId;
            button.setAttribute('aria-label', action.text);

            button.addEventListener('click', () => {
                this.handleQuickAction(action.faqId);
            });

            this.elements.quickActions.appendChild(button);
        });

        console.log(`✅ ${actions.length} acciones rápidas renderizadas`);
    },

    // ============================================
    // DETECCIÓN DE PÁGINA
    // ============================================
    detectPage: function() {
        const path = window.location.pathname.toLowerCase();

        if (path.includes('admision')) return 'admision';
        if (path.includes('contacto')) return 'contacto';

        return 'default';
    },

    // ============================================
    // MENSAJE DE BIENVENIDA
    // ============================================
    loadWelcomeMessage: function() {
        // Si ya tiene un rol seleccionado, mostrar mensaje normal
        if (this.state.userRole) {
            const page = this.detectPage();
            const welcomeMsg = CHATBOT_DATA.welcomeMessages[page] || CHATBOT_DATA.welcomeMessages.default;
            setTimeout(() => {
                this.addMessage(welcomeMsg, 'bot');
            }, 500);
            return;
        }

        // Si no tiene rol, mostrar selección de roles
        setTimeout(() => {
            this.showRoleSelection();
        }, 500);
    },

    // ============================================
    // SELECCIÓN DE ROL
    // ============================================
    showRoleSelection: function() {
        const welcomeMsg = '¡Hola! Bienvenido al Asistente Virtual de la UNSM 👋\n\nPara brindarte la mejor ayuda, por favor indícame tu rol:';

        // Añadir mensaje de bienvenida
        this.addMessage(welcomeMsg, 'bot');

        // Crear botones de rol
        setTimeout(() => {
            this.addRoleButtons();
        }, 300);
    },

    addRoleButtons: function() {
        const messageDiv = document.createElement('div');
        messageDiv.className = 'chatbot__message chatbot__message--bot chatbot__role-selection';

        const rolesContainer = document.createElement('div');
        rolesContainer.className = 'chatbot__roles-container';

        // Crear un botón para cada rol
        for (const roleKey in CHATBOT_DATA.roles) {
            const role = CHATBOT_DATA.roles[roleKey];
            const button = document.createElement('button');
            button.className = 'chatbot__role-button';
            button.dataset.role = role.id;
            button.innerHTML = `
                <span class="chatbot__role-icon">${role.icon}</span>
                <div class="chatbot__role-info">
                    <span class="chatbot__role-name">${role.name}</span>
                    <span class="chatbot__role-desc">${role.description}</span>
                </div>
            `;

            button.addEventListener('click', () => {
                this.selectRole(role.id);
            });

            rolesContainer.appendChild(button);
        }

        messageDiv.appendChild(rolesContainer);
        this.elements.messages.appendChild(messageDiv);
        this.scrollToBottom();
    },

    selectRole: function(roleId) {
        // Guardar el rol seleccionado (solo en memoria, no persistente)
        this.state.userRole = roleId;

        // Obtener información del rol
        const role = CHATBOT_DATA.roles[roleId];

        // Añadir mensaje del usuario seleccionando el rol
        this.addMessage(`${role.icon} ${role.name}`, 'user');

        // Remover los botones de rol
        const roleSelection = document.querySelector('.chatbot__role-selection');
        if (roleSelection) {
            roleSelection.style.opacity = '0.5';
            roleSelection.style.pointerEvents = 'none';
        }

        // Mostrar confirmación y actualizar acciones rápidas
        setTimeout(() => {
            const confirmMsg = `¡Perfecto! Te ayudaré como ${role.name.toLowerCase()}.\n\n¿En qué puedo ayudarte hoy?`;
            this.addMessage(confirmMsg, 'bot');

            // Actualizar y mostrar acciones rápidas según el rol
            setTimeout(() => {
                // Asegurarse de que las acciones rápidas estén visibles
                if (this.elements.quickActions) {
                    this.elements.quickActions.style.display = 'flex';
                }
                this.setQuickActions();
            }, 200);
        }, 800);

        console.log('✅ Rol seleccionado:', roleId);
    },

    // Función para cambiar de rol
    changeRole: function() {
        // Limpiar el rol actual
        this.state.userRole = null;

        // Ocultar acciones rápidas
        if (this.elements.quickActions) {
            this.elements.quickActions.style.display = 'none';
        }

        // Mostrar mensaje y opciones de rol
        setTimeout(() => {
            this.addMessage('Claro, selecciona tu nuevo rol:', 'bot');
            setTimeout(() => {
                this.addRoleButtons();
            }, 300);
        }, 500);
    },

    // ============================================
    // INDICADOR DE ESCRITURA (TYPING)
    // ============================================
    showTyping: function() {
        this.elements.typing.style.display = 'flex';
        this.state.isTyping = true;
        this.scrollToBottom();
    },

    hideTyping: function() {
        this.elements.typing.style.display = 'none';
        this.state.isTyping = false;
    },

    // ============================================
    // UTILIDADES
    // ============================================
    findFaqById: function(id) {
        for (const category of Object.values(CHATBOT_DATA.categories)) {
            const faq = category.faqs.find(f => f.id === id);
            if (faq) return faq;
        }
        return null;
    },

    formatTimestamp: function(date) {
        return date.toLocaleTimeString('es-PE', {
            hour: '2-digit',
            minute: '2-digit'
        });
    },

    scrollToBottom: function() {
        setTimeout(() => {
            this.elements.messages.scrollTop = this.elements.messages.scrollHeight;
        }, 100);
    },

    // ============================================
    // SANITIZACIÓN Y FORMATO (SEGURIDAD)
    // ============================================
    sanitizeHTML: function(text) {
        const temp = document.createElement('div');
        temp.textContent = text;
        return temp.innerHTML;
    },

    formatMessageText: function(text) {
        // Primero sanitizar para evitar XSS
        const sanitized = this.sanitizeHTML(text);
        // Luego convertir saltos de línea a <br>
        return sanitized.replace(/\n/g, '<br>');
    }
};

// ============================================
// AUTO-INICIALIZACIÓN
// ============================================
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function() {
        // Esperar a que UNSM esté disponible
        if (typeof UNSM !== 'undefined') {
            UNSM.Chatbot.init();
        } else {
            console.error('Namespace UNSM no encontrado. Asegúrate de cargar script.js primero.');
        }
    });
} else {
    // DOM ya está listo
    if (typeof UNSM !== 'undefined') {
        UNSM.Chatbot.init();
    } else {
        console.error('Namespace UNSM no encontrado. Asegúrate de cargar script.js primero.');
    }
}
