-- ==========================================
-- Script de Creación de Base de Datos
-- Sistema de Autenticación - UNSM
-- ==========================================

-- Crear base de datos
CREATE DATABASE IF NOT EXISTS unsm_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

USE unsm_db;

-- ==========================================
-- Tabla de Usuarios
-- ==========================================
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL,
    email VARCHAR(100) NOT NULL UNIQUE,
    first_name VARCHAR(50) NOT NULL,
    last_name VARCHAR(50) NOT NULL,
    role ENUM('admin', 'docente', 'estudiante') DEFAULT 'estudiante',
    active TINYINT(1) DEFAULT 1,
    last_login DATETIME NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    INDEX idx_username (username),
    INDEX idx_email (email),
    INDEX idx_role (role),
    INDEX idx_active (active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ==========================================
-- Tabla de Logs de Acceso
-- ==========================================
CREATE TABLE IF NOT EXISTS login_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    ip_address VARCHAR(45) NOT NULL,
    user_agent VARCHAR(255) NULL,
    success TINYINT(1) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_id (user_id),
    INDEX idx_created_at (created_at),
    INDEX idx_success (success)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ==========================================
-- Tabla de Tokens de Recuperación de Contraseña
-- ==========================================
CREATE TABLE IF NOT EXISTS password_reset_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    token VARCHAR(64) NOT NULL UNIQUE,
    expires_at DATETIME NOT NULL,
    used TINYINT(1) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_token (token),
    INDEX idx_user_id (user_id),
    INDEX idx_expires_at (expires_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ==========================================
-- Datos de Ejemplo (Usuario Administrador)
-- ==========================================
-- Contraseña por defecto: admin123 (IMPORTANTE: Cambiar en producción)
INSERT INTO users (username, password, email, first_name, last_name, role, active) VALUES
('admin', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'admin@unsm.edu.pe', 'Administrador', 'Sistema', 'admin', 1),
('docente1', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'docente1@unsm.edu.pe', 'Juan', 'Pérez', 'docente', 1),
('estudiante1', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 'estudiante1@unsm.edu.pe', 'María', 'García', 'estudiante', 1);

-- ==========================================
-- NOTAS IMPORTANTES
-- ==========================================
-- 1. La contraseña por defecto para todos los usuarios es: admin123
-- 2. El hash es generado con password_hash() de PHP usando PASSWORD_DEFAULT
-- 3. IMPORTANTE: Cambiar todas las contraseñas en producción
-- 4. Para generar un nuevo hash de contraseña en PHP usar:
--    password_hash('tu_contraseña', PASSWORD_DEFAULT);

-- ==========================================
-- Vista para consultar usuarios activos
-- ==========================================
CREATE OR REPLACE VIEW active_users AS
SELECT
    id,
    username,
    email,
    CONCAT(first_name, ' ', last_name) as full_name,
    role,
    last_login,
    created_at
FROM users
WHERE active = 1;

-- ==========================================
-- Procedimiento para desactivar usuario
-- ==========================================
DELIMITER //

CREATE PROCEDURE deactivate_user(IN p_user_id INT)
BEGIN
    UPDATE users SET active = 0, updated_at = CURRENT_TIMESTAMP WHERE id = p_user_id;
END //

CREATE PROCEDURE activate_user(IN p_user_id INT)
BEGIN
    UPDATE users SET active = 1, updated_at = CURRENT_TIMESTAMP WHERE id = p_user_id;
END //

DELIMITER ;

-- ==========================================
-- VERIFICACIÓN FINAL
-- ==========================================
SHOW TABLES;
SELECT * FROM users;
