<?php
/**
 * Sistema de Autenticación
 * Universidad Nacional de San Martín - UNSM
 */

/**
 * Autentica un usuario contra la base de datos
 * @param string $username Usuario
 * @param string $password Contraseña
 * @return array Resultado de la autenticación
 */
function authenticate_user($username, $password) {
    // Cargar configuración de base de datos
    require_once __DIR__ . '/db_config.php';

    try {
        $conn = get_db_connection();

        // Preparar consulta SQL para prevenir inyección SQL
        $stmt = $conn->prepare("SELECT id, username, password, email, role, active FROM users WHERE username = ? AND active = 1 LIMIT 1");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        $result = $stmt->get_result();

        if ($result->num_rows === 0) {
            // Usuario no encontrado o inactivo
            return [
                'success' => false,
                'message' => 'Usuario o contraseña incorrectos.'
            ];
        }

        $user = $result->fetch_assoc();

        // Verificar contraseña
        if (password_verify($password, $user['password'])) {
            // Autenticación exitosa
            // Actualizar último acceso
            update_last_login($conn, $user['id']);

            // Registrar en log de accesos
            log_login_attempt($conn, $user['id'], true);

            $stmt->close();
            $conn->close();

            return [
                'success' => true,
                'user_id' => $user['id'],
                'username' => $user['username'],
                'email' => $user['email'],
                'role' => $user['role']
            ];
        } else {
            // Contraseña incorrecta
            log_login_attempt($conn, $user['id'], false);

            $stmt->close();
            $conn->close();

            return [
                'success' => false,
                'message' => 'Usuario o contraseña incorrectos.'
            ];
        }
    } catch (Exception $e) {
        error_log("Error en autenticación: " . $e->getMessage());
        return [
            'success' => false,
            'message' => 'Error en el sistema. Por favor, intente más tarde.'
        ];
    }
}

/**
 * Actualiza la fecha de último acceso del usuario
 */
function update_last_login($conn, $user_id) {
    $stmt = $conn->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
    $stmt->bind_param("i", $user_id);
    $stmt->execute();
    $stmt->close();
}

/**
 * Registra intentos de inicio de sesión
 */
function log_login_attempt($conn, $user_id, $success) {
    $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';

    $stmt = $conn->prepare("INSERT INTO login_logs (user_id, ip_address, user_agent, success, created_at) VALUES (?, ?, ?, ?, NOW())");
    $stmt->bind_param("issi", $user_id, $ip_address, $user_agent, $success);
    $stmt->execute();
    $stmt->close();
}

/**
 * Verifica si un usuario está autenticado
 */
function is_authenticated() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    return isset($_SESSION['user_logged_in']) && $_SESSION['user_logged_in'] === true;
}

/**
 * Obtiene el ID del usuario actual
 */
function get_current_user_id() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    return $_SESSION['user_id'] ?? null;
}

/**
 * Obtiene el rol del usuario actual
 */
function get_current_user_role() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    return $_SESSION['user_role'] ?? null;
}

/**
 * Cierra la sesión del usuario
 */
function logout_user() {
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }

    // Destruir todas las variables de sesión
    $_SESSION = array();

    // Destruir la cookie de sesión
    if (isset($_COOKIE[session_name()])) {
        setcookie(session_name(), '', time() - 3600, '/');
    }

    // Destruir la sesión
    session_destroy();
}

/**
 * Requiere autenticación - Redirige al login si no está autenticado
 */
function require_authentication() {
    if (!is_authenticated()) {
        header('Location: /login.php');
        exit;
    }
}

/**
 * Verifica si el usuario tiene un rol específico
 */
function has_role($required_role) {
    $user_role = get_current_user_role();

    // Roles jerárquicos: admin > docente > estudiante
    $role_hierarchy = [
        'admin' => 3,
        'docente' => 2,
        'estudiante' => 1
    ];

    $user_level = $role_hierarchy[$user_role] ?? 0;
    $required_level = $role_hierarchy[$required_role] ?? 0;

    return $user_level >= $required_level;
}
