/**
 * Script para la funcionalidad de tabs en la página de Admisión UNSM
 */

document.addEventListener('DOMContentLoaded', function() {
    // Seleccionar todos los botones de tabs
    const tabButtons = document.querySelectorAll('.tab-button');
    const tabContents = document.querySelectorAll('.tab-content');

    // ============================================
    // FUNCIONALIDAD DE ACORDEÓN PARA LAS SEDES
    // ============================================
    function initializeAccordion() {
        const sedeCards = document.querySelectorAll('.sede-card.collapsible');
        console.log('Inicializando acordeón, sedes encontradas:', sedeCards.length);

        sedeCards.forEach((card, index) => {
            const header = card.querySelector('.sede-header');
            const toggle = card.querySelector('.sede-toggle');

            // Remover event listeners previos (si existen)
            const newHeader = header.cloneNode(true);
            header.parentNode.replaceChild(newHeader, header);

            // Hacer clic en el header o el botón toggle
            newHeader.addEventListener('click', function(e) {
                e.preventDefault();
                e.stopPropagation();

                console.log('Click en sede:', card.querySelector('h3').textContent);

                // Toggle la clase active
                const wasActive = card.classList.contains('active');

                // Cerrar todas las sedes primero
                sedeCards.forEach(otherCard => {
                    otherCard.classList.remove('active');
                });

                // Si no estaba activa, activarla
                if (!wasActive) {
                    card.classList.add('active');
                    console.log('Sede abierta');
                } else {
                    console.log('Sede cerrada');
                }
            });
        });
    }

    // Inicializar acordeón al cargar
    initializeAccordion();

    // Función para cambiar de tab
    function switchTab(tabId) {
        // Remover clase active de todos los botones y contenidos
        tabButtons.forEach(button => {
            button.classList.remove('active');
        });

        tabContents.forEach(content => {
            content.classList.remove('active');
        });

        // Agregar clase active al botón y contenido seleccionado
        const selectedButton = document.querySelector(`[data-tab="${tabId}"]`);
        const selectedContent = document.getElementById(tabId);

        if (selectedButton && selectedContent) {
            selectedButton.classList.add('active');
            selectedContent.classList.add('active');

            // Reinicializar acordeón después de cambiar de tab
            setTimeout(function() {
                initializeAccordion();
            }, 100);

            // Hacer scroll suave hacia arriba del contenido
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
    }

    // Agregar event listeners a cada botón de tab
    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const tabId = this.getAttribute('data-tab');
            switchTab(tabId);
        });
    });

    // Manejar tabs desde URL hash
    if (window.location.hash) {
        const tabFromHash = window.location.hash.substring(1);
        if (document.getElementById(tabFromHash)) {
            switchTab(tabFromHash);
        }
    }

    // Actualizar URL hash cuando cambia el tab
    tabButtons.forEach(button => {
        button.addEventListener('click', function() {
            const tabId = this.getAttribute('data-tab');
            history.pushState(null, null, `#${tabId}`);
        });
    });

    // Animación suave para hover en las tarjetas de sedes
    const sedeCards = document.querySelectorAll('.sede-card');
    sedeCards.forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transition = 'transform 0.3s ease, box-shadow 0.3s ease';
        });
    });

    // Animación para el botón de postular
    const btnPostula = document.querySelector('.btn-postula');
    if (btnPostula) {
        btnPostula.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-2px) scale(1.02)';
        });

        btnPostula.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
        });
    }

    // Smooth scroll para links internos
    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', function (e) {
            const href = this.getAttribute('href');
            if (href !== '#' && href.length > 1) {
                e.preventDefault();
                const target = document.querySelector(href);
                if (target) {
                    target.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            }
        });
    });

    // Destacar tab activo al hacer scroll (opcional)
    const observerOptions = {
        root: null,
        rootMargin: '0px',
        threshold: 0.5
    };

    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const tabId = entry.target.id;
                if (tabId) {
                    // No cambiar automáticamente el tab activo, solo si el usuario hace clic
                }
            }
        });
    }, observerOptions);

    // Observar todos los contenidos de tabs
    tabContents.forEach(content => {
        observer.observe(content);
    });

    // Lazy loading para imágenes (si las hay)
    if ('IntersectionObserver' in window) {
        const imageObserver = new IntersectionObserver((entries, observer) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const img = entry.target;
                    if (img.dataset.src) {
                        img.src = img.dataset.src;
                        img.removeAttribute('data-src');
                        imageObserver.unobserve(img);
                    }
                }
            });
        });

        document.querySelectorAll('img[data-src]').forEach(img => {
            imageObserver.observe(img);
        });
    }

    // Animación de entrada para las tarjetas de modalidades
    const modalidadItems = document.querySelectorAll('.modalidad-item');
    const faqItems = document.querySelectorAll('.faq-item');

    const itemObserver = new IntersectionObserver((entries) => {
        entries.forEach((entry, index) => {
            if (entry.isIntersecting) {
                setTimeout(() => {
                    entry.target.style.opacity = '1';
                    entry.target.style.transform = 'translateX(0)';
                }, index * 100);
                itemObserver.unobserve(entry.target);
            }
        });
    }, {
        threshold: 0.1
    });

    modalidadItems.forEach(item => {
        item.style.opacity = '0';
        item.style.transform = 'translateX(-20px)';
        item.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
        itemObserver.observe(item);
    });

    faqItems.forEach(item => {
        item.style.opacity = '0';
        item.style.transform = 'translateX(-20px)';
        item.style.transition = 'opacity 0.5s ease, transform 0.5s ease';
        itemObserver.observe(item);
    });

    // Función para actualizar la fecha de eventos (opcional)
    function updateEventDates() {
        const eventoFechas = document.querySelectorAll('.evento-fecha');
        // Aquí puedes agregar lógica para actualizar fechas dinámicamente si es necesario
    }

    // Contador de caracteres para formularios (si se implementan en el futuro)
    const textareas = document.querySelectorAll('textarea[maxlength]');
    textareas.forEach(textarea => {
        const maxLength = textarea.getAttribute('maxlength');
        const counter = document.createElement('div');
        counter.className = 'char-counter';
        counter.textContent = `0 / ${maxLength}`;
        textarea.parentNode.insertBefore(counter, textarea.nextSibling);

        textarea.addEventListener('input', function() {
            const currentLength = this.value.length;
            counter.textContent = `${currentLength} / ${maxLength}`;

            if (currentLength >= maxLength * 0.9) {
                counter.style.color = '#D32F2F';
            } else {
                counter.style.color = '#666';
            }
        });
    });

    // Validación básica para el botón de postular (redirigir a página de inscripción)
    if (btnPostula) {
        btnPostula.addEventListener('click', function(e) {
            e.preventDefault();
            // Aquí puedes agregar la lógica de redirección o modal
            alert('Serás redirigido a la plataforma de inscripción.');
            // window.location.href = 'url_de_inscripcion';
        });
    }

    // Imprimir información de la sección actual en consola (para debugging)
    console.log('Script de Admisión UNSM cargado correctamente');
    console.log(`Total de tabs: ${tabButtons.length}`);
    console.log(`Total de contenidos: ${tabContents.length}`);
});

// Función para compartir en redes sociales (opcional)
function shareOnSocial(platform, url, text) {
    const shareUrls = {
        facebook: `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(url)}`,
        twitter: `https://twitter.com/intent/tweet?url=${encodeURIComponent(url)}&text=${encodeURIComponent(text)}`,
        whatsapp: `https://wa.me/?text=${encodeURIComponent(text + ' ' + url)}`,
        linkedin: `https://www.linkedin.com/shareArticle?mini=true&url=${encodeURIComponent(url)}&title=${encodeURIComponent(text)}`
    };

    if (shareUrls[platform]) {
        window.open(shareUrls[platform], '_blank', 'width=600,height=400');
    }
}

// Exportar funciones si es necesario
if (typeof module !== 'undefined' && module.exports) {
    module.exports = {
        shareOnSocial
    };
}
